#!/usr/bin/env bash
# update_tri.sh — usage: ./update_tri.sh domain.com
# Updates ads.js & redirect.json from update_tri.zip, backing up existing files.

set -euo pipefail

if [[ $# -ne 1 ]]; then
  echo "Usage: $0 domain.com"
  exit 1
fi

DOMAIN="$1"
WEBROOT="/var/www/html"
TARGET="${WEBROOT}/${DOMAIN}"
ZIP_URL="https://cdn.jasaorder.com/agc2025/ads2/update_tri.zip"
ZIP_NAME="update_tri.zip"

# Basic checks
for cmd in wget unzip; do
  if ! command -v "$cmd" >/dev/null 2>&1; then
    echo "Error: '$cmd' is not installed. Please install it and re-run." >&2
    exit 1
  fi
done

if [[ ! -d "$TARGET" ]]; then
  echo "Error: Target directory '$TARGET' does not exist." >&2
  exit 1
fi

cd "$TARGET"

# Helper: backup if file exists (timestamped to avoid overwrite)
backup_if_exists () {
  local f="$1"
  if [[ -f "$f" ]]; then
    local ts
    ts="$(date +%Y%m%d-%H%M%S)"
    mv -v "$f" "${f}.${ts}.backup"
  else
    echo "Info: '$f' not found, skipping backup."
  fi
}

# Backups (as requested)
backup_if_exists "ads.js"
backup_if_exists "redirect.json"
backup_if_exists "setup.php"

# Fetch and extract update
rm -f "$ZIP_NAME"
echo "Downloading: $ZIP_URL"
wget -q "$ZIP_URL" -O "$ZIP_NAME"

echo "Unzipping: $ZIP_NAME"
unzip -o "$ZIP_NAME" >/dev/null
rm -f "$ZIP_NAME"

# Permissions (as requested)
if [[ -f "redirect.json" ]]; then
  chmod 777 "redirect.json"
else
  echo "Warning: redirect.json not found after update."
fi

if [[ -f "ads.js" ]]; then
  chmod 777 "ads.js"
else
  echo "Warning: ads.js not found after update."
fi

echo "Update completed for $DOMAIN."

