#!/usr/bin/env bash
# Usage: sudo ./create.sh domain.com email@example.com
# Creates /var/www/html/<domain>, downloads package, sets perms, and updates email inside ssl.sh.

set -euo pipefail

if [[ $# -ne 2 ]]; then
  echo "Usage: $0 domain.com email@example.com"
  exit 1
fi

if [[ $(id -u) -ne 0 ]]; then
  echo "Please run as root (sudo)."
  exit 1
fi

DOMAIN="${1%/}"
EMAIL="$2"
ROOT="/var/www/html"
SITE_DIR="$ROOT/$DOMAIN"

# Dependencies
for cmd in wget unzip chown chmod mkdir sed; do
  command -v "$cmd" >/dev/null 2>&1 || { echo "Missing dependency: $cmd"; exit 1; }
done
getent passwd www-data >/dev/null || { echo "System user 'www-data' not found."; exit 1; }

echo "[*] Creating site directory: $SITE_DIR"
mkdir -p "$SITE_DIR"
chmod 777 "$SITE_DIR"   # per request

cd "$SITE_DIR"

ZIP="agc_update.zip"
URL="https://install.cdnweb.info/agc_full/agc_update.zip?nocache=$(date +%s)"

echo "[*] Downloading package…"
wget -q "$URL" -O "$ZIP"

echo "[*] Unzipping…"
unzip -o "$ZIP" >/dev/null
rm -f "$ZIP"

echo "[*] Setting ownership/permissions…"
[[ -d db ]] && chown -R www-data:www-data db
[[ -f db/dbPost.db ]] && chown www-data:www-data db/dbPost.db

for p in logs uploads assets assets/prompt ads; do
  [[ -e "$p" ]] && { chown -R www-data:www-data "$p"; chmod -R 755 "$p"; }
done

[[ -f ssl.sh ]] && chmod +x ssl.sh
[[ -d config ]] && chmod 777 config  # per request

# --- Replace email inside ssl.sh ---
if [[ -f ssl.sh ]]; then
  # Escape / and & in email for sed replacement
  EMAIL_ESC="${EMAIL//\//\\/}"
  EMAIL_ESC="${EMAIL_ESC//&/\\&}"

  # If a known default email exists, replace it with user-supplied email
  if grep -qE 'amin\.uzan@gmail\.com|you@example\.com' ssl.sh; then
    sed -i -E "s/(amin\.uzan@gmail\.com|you@example\.com)/${EMAIL_ESC}/g" ssl.sh
  fi

  # Replace values passed to certbot flags or EMAIL= assignments (best-effort)
  sed -i -E "s/(--email|-m)[ =]\"?[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\.[A-Za-z]{2,}\"?/\1 ${EMAIL_ESC}/g" ssl.sh || true
  sed -i -E "s/(EMAIL=)\"?[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\.[A-Za-z]{2,}\"?/\1${EMAIL_ESC}/g" ssl.sh || true

  # Fallback: if none of the above matched, replace the first email-like token
  if ! grep -q "$EMAIL" ssl.sh; then
    sed -i -E "0,/[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\.[A-Za-z]{2,}/{s//${EMAIL_ESC}/}" ssl.sh
  fi

  echo "[*] Updated email in ssl.sh to ${EMAIL}"
else
  echo "[!] ssl.sh not found; skipped email update."
fi

echo "[✓] Done for $DOMAIN in $SITE_DIR"

